<?php
// ---------- Safe prelude (กัน variable undefined) ----------
$BASE   = $BASE   ?? '';
$seed   = is_array($seed ?? null) ? $seed : [];
$projects = is_array($projects ?? null) ? $projects : ($seed['projects'] ?? []);
$jobs     = is_array($jobs ?? null)     ? $jobs     : ($seed['jobs'] ?? []);

// fallback helpers (ถ้ายังไม่ได้ประกาศจากระบบหลัก)
if (!function_exists('can_view_all')) { function can_view_all(){ return true; } }
if (!function_exists('user_team'))    { function user_team(){ return 'General'; } }
if (!function_exists('can_plan'))     { function can_plan(){ return true; } }

// index งานต่อโปรเจกต์ (ปลอดภัยแม้ $jobs ว่าง)
$jobsByProject = [];
foreach ($jobs as $j) {
  $pid = (int)($j['project_id'] ?? 0);
  if ($pid > 0) $jobsByProject[$pid][] = $j;
}

// คำนวณ % ความคืบหน้าแบบกันพัง
if (!function_exists('project_percent')) {
  function project_percent(array $p, array $jobsByProject): int {
    if (isset($p['progress'])) {
      $x = (int)$p['progress'];
      return max(0, min(100, $x));
    }
    $pid   = (int)($p['id'] ?? 0);
    $items = $jobsByProject[$pid] ?? [];
    if (!$items) return 0;

    $total = 0.0; $done = 0.0;
    foreach ($items as $t) {
      $w  = isset($t['plan_hours']) ? max(1.0, (float)$t['plan_hours']) : 1.0;
      $st = strtolower((string)($t['status'] ?? ''));
      $tp = isset($t['progress']) ? max(0.0, min(100.0, (float)$t['progress']))
           : (($st==='done'||$st==='completed') ? 100.0 : ($st==='active' ? 50.0 : ($st==='planning' ? 10.0 : 0.0)));
      $total += $w; $done += $w * ($tp/100.0);
    }
    return (int)round(100.0 * $done / max(1.0, $total));
  }
}

// filter ตามทีมถ้าไม่มีสิทธิ์ดูทั้งหมด
if (!can_view_all()) {
  $team = user_team();
  $projects = array_values(array_filter($projects, fn($p)=> ($p['team'] ?? '') === $team));
}

// รายชื่อทีมสำหรับ filter/datalist
$teamSet = [];
foreach ($projects as $p) {
  $t = trim((string)($p['team'] ?? ''));
  if ($t !== '') $teamSet[$t] = 1;
}
?>

<section class="section page"><!-- .page ช่วยดัน footer ลงล่าง -->
  <div class="section-head">
    <div class="lhs">
      <h2>โปรเจกต์</h2>
      <span class="count muted" id="proj-count"></span>
    </div>
    <div class="toolbar">
      <div class="field">
        <input id="q" type="search" placeholder="ค้นหา: ชื่อ / โค้ด / ทีม…">
      </div>
      <div class="field">
        <select id="filter-team">
          <option value="">ทุกทีม</option>
          <?php foreach (array_keys($teamSet) as $t): ?>
            <option value="<?= htmlspecialchars($t) ?>"><?= htmlspecialchars($t) ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="field">
        <select id="filter-status">
          <option value="">ทุกสถานะ</option>
          <option value="planning">Planning</option>
          <option value="active">Active</option>
          <option value="done">Done</option>
          <option value="on hold">On Hold</option>
        </select>
      </div>
      <div class="field">
        <select id="sort">
          <option value="recent">เรียง: ล่าสุดก่อน</option>
          <option value="name">เรียง: ชื่อ (A–Z)</option>
          <option value="team">เรียง: ทีม</option>
          <option value="progress">เรียง: ความคืบหน้า</option>
        </select>
      </div>
      <?php if (can_plan()): ?>
        <button class="btn primary" onclick="openProjectModal()">+ เพิ่มโปรเจกต์</button>
      <?php endif; ?>
    </div>
  </div>

  <div id="project-empty" class="empty hidden">
    <div class="emoji">🗂️</div>
    <div class="title">ยังไม่มีโปรเจกต์ที่ตรงเงื่อนไข</div>
    <div class="hint">ลองล้างตัวกรองหรือค้นหาด้วยคำอื่นดูนะ</div>
  </div>

  <div class="cards" id="project-cards">
    <?php foreach ($projects as $p): ?>
      <?php
        $pct = project_percent($p, $jobsByProject);
        $statusRaw   = trim((string)($p['status'] ?? ''));
        $statusData  = strtolower($statusRaw);              // สำหรับค้นหา/กรอง
        $statusClass = preg_replace('/\s+/', '-', strtolower($statusRaw)); // สำหรับชื่อ class CSS
      ?>
      <a
        class="card project"
        href="<?= $BASE ?>/project?id=<?= (int)($p['id'] ?? 0) ?>"
        data-name="<?= htmlspecialchars(strtolower($p['name'] ?? '')) ?>"
        data-code="<?= htmlspecialchars(strtolower($p['code'] ?? '')) ?>"
        data-team="<?= htmlspecialchars(strtolower($p['team'] ?? '')) ?>"
        data-status="<?= htmlspecialchars($statusData) ?>"
        data-progress="<?= (int)$pct ?>"
        data-updated="<?= htmlspecialchars($p['updated_at'] ?? '') ?>"
      >
        <div class="toprow">
          <div class="title ellip"><?= htmlspecialchars($p['name'] ?? 'Untitled') ?></div>
          <?php if ($statusRaw !== ''): ?>
            <span class="pill <?= $statusClass ?>"><?= htmlspecialchars($statusRaw) ?></span>
          <?php endif; ?>
        </div>

        <div class="meta-row">
          <span class="chip"><b>#<?= htmlspecialchars($p['code'] ?? '-') ?></b></span>
          <span class="chip"><?= htmlspecialchars($p['team'] ?? '-') ?></span>
        </div>

        <div class="progress-wrap">
          <div class="progress"><div class="progress__bar" style="--pct: <?= $pct ?>%"></div></div>
          <span class="progress__pct"><?= $pct ?>%</span>
        </div>
      </a>
    <?php endforeach; ?>
  </div>
</section>

<?php if (can_plan()): ?>
<!-- Modal -->
<div id="project-modal" class="modal hidden" role="dialog" aria-modal="true" aria-labelledby="modal-title">
  <div class="modal-card">
    <div class="modal-head">
      <div class="modal-title" id="modal-title">สร้างโปรเจกต์ใหม่</div>
      <button class="iconbtn" onclick="closeProjectModal()" aria-label="ปิด">✕</button>
    </div>
    <div class="modal-body grid">
      <label>ชื่อโปรเจกต์
        <input type="text" id="p-name" placeholder="เช่น ระบบ Smart Cabinet Phase 2">
      </label>
      <label>รหัสโปรเจกต์
        <input type="text" id="p-code" placeholder="PRJ-XXX-001">
      </label>
      <label>ทีมงาน
        <input type="text" id="p-team" list="teams-datalist" placeholder="เลือกหรือพิมพ์ทีม">
        <datalist id="teams-datalist">
          <?php foreach (array_keys($teamSet) as $t): ?>
            <option value="<?= htmlspecialchars($t) ?>"></option>
          <?php endforeach; ?>
        </datalist>
      </label>
      <label>สถานะ
        <select id="p-status">
          <option>Planning</option>
          <option selected>Active</option>
          <option>Done</option>
          <option>On Hold</option>
        </select>
      </label>
    </div>
    <div class="modal-foot">
      <button class="btn" onclick="closeProjectModal()">ยกเลิก</button>
      <button class="btn primary" onclick="saveProjectMock()">บันทึก (Mock)</button>
    </div>
  </div>
</div>
<?php endif; ?>

<style>
/* ===== Light Theme (โทนขาว) ===== */
:root{
  --bg:#ffffff;         /* พื้นหลังเพจ */
  --card:#ffffff;       /* พื้นหลังการ์ด */
  --line:#e5e7eb;       /* เส้น/กรอบ */
  --fg:#0f172a;         /* ตัวอักษรหลัก */
  --muted:#64748b;      /* ตัวอักษรรอง */
  --chip:#f1f5f9;       /* พื้น chip */
  --brand:#0ea5e9;      /* ฟ้า */
  --brand-2:#22d3ee;    /* ฟ้าอ่อน/เขียวฟ้า */
}
html,body{ height:100%; }
body{
  min-height:100vh;
  display:flex; flex-direction:column;   /* ดัน footer ลงล่างเสมอ */
  background:var(--bg); color:var(--fg);
}

/* คอนเทนต์หลัก: ใช้คลาส .page ที่หุ้ม section นี้แล้ว */
.page, .main, .app, .container, .wrap { flex:1 0 auto; }

/* ===== Layout / Toolbar / Buttons ===== */
.section{ padding:18px 0; }
.section-head{ display:flex; flex-wrap:wrap; gap:12px; align-items:center; justify-content:space-between; margin-bottom:14px; }
.section-head .lhs{ display:flex; align-items:center; gap:10px; }
.section-head h2{ margin:0; font-size:20px; }
.count{ font-size:13px; color:var(--muted); }

.toolbar{ display:flex; flex-wrap:wrap; gap:10px; align-items:center; }
.field input[type="search"], .field select{
  height:36px; padding:0 12px; border-radius:10px; outline:none;
  border:1px solid var(--line); background:#fff; color:var(--fg);
}
.field input::placeholder{ color:#94a3b8; }

.btn{
  display:inline-flex; align-items:center; gap:8px; height:36px; padding:0 14px; border-radius:10px;
  font-size:14px; font-weight:700; cursor:pointer; border:1px solid var(--line); background:#fff; color:var(--fg);
  transition: filter .15s ease, box-shadow .15s ease;
}
.btn:hover{ filter:brightness(0.98); box-shadow:0 3px 10px rgba(2,6,23,.08); }
.btn.primary{
  background:linear-gradient(90deg,var(--brand),var(--brand-2)); border-color:transparent; color:#fff;
}

/* ===== Grid & Empty ===== */
.cards{
  display:grid;
  gap:12px;
  grid-template-columns: repeat(4, 1fr);
}
.empty{ border:1px dashed var(--line); border-radius:14px; padding:24px; text-align:center; background:#fafafa; color:var(--fg); }
.empty.hidden{ display:none; }
.empty .emoji{ font-size:28px; margin-bottom:6px; }
.empty .title{ font-weight:700; }
.empty .hint{ color:var(--muted); font-size:13px; margin-top:4px; }

/* ===== Card ===== */
.card.project{
  display:flex; flex-direction:column; gap:10px; min-height:148px; padding:14px; border-radius:16px;
  background:var(--card); border:1px solid var(--line); color:inherit; text-decoration:none;
  box-shadow:0 3px 12px rgba(2,6,23,.06);
  transition: transform .14s ease, box-shadow .14s ease, border-color .14s ease;
}
.card.project:hover{
  transform:translateY(-2px); box-shadow:0 12px 24px rgba(2,6,23,.10); border-color:#d0d7e2;
}
.toprow{ display:flex; align-items:center; justify-content:space-between; gap:8px; }
.title{ font-size:16px; font-weight:800; }
.ellip{ white-space:nowrap; overflow:hidden; text-overflow:ellipsis; }

.meta-row{ display:flex; gap:8px; flex-wrap:wrap; }
.chip{
  display:inline-flex; align-items:center; gap:6px; height:24px; padding:0 10px; border-radius:999px; font-size:12px;
  background:var(--chip); border:1px solid var(--line); color:#0b2540;
}

/* ป้ายสถานะ (light) */
.pill{
  font-size:12px; font-weight:800; padding:4px 10px; border-radius:999px; border:1px solid var(--line);
  background:#f8fafc; color:#0b2540;
}
.pill.planning{ background:#eef2ff; color:#3730a3; border-color:#c7d2fe; }
.pill.active{ background:#ecfeff; color:#0f766e; border-color:#a5f3fc; }
.pill.done{ background:#ecfdf5; color:#166534; border-color:#bbf7d0; }
.pill.on-hold{ background:#fffbeb; color:#92400e; border-color:#fde68a; } /* ใช้ขีดกลาง */

/* ===== Progress ===== */
.progress-wrap{ display:flex; align-items:center; gap:10px; margin-top:auto; }
.progress{ height:8px; flex:1; border-radius:999px; background:#eef2f7; overflow:hidden; }
.progress__bar{ height:100%; width:var(--pct); background:linear-gradient(90deg,var(--brand),var(--brand-2)); transition:width .25s ease; }
.progress__pct{ min-width:42px; text-align:right; font-weight:800; }

/* ===== Modal ===== */
.modal{ position:fixed; inset:0; display:grid; place-items:center; background:rgba(2,6,23,.25); backdrop-filter: blur(2px); z-index:1000; }
.modal.hidden{ display:none; }
.modal-card{ width:min(720px, 94vw); border-radius:18px; overflow:hidden; background:#fff; color:var(--fg); border:1px solid var(--line); box-shadow:0 20px 60px rgba(2,6,23,.18); animation:pop .14s ease-out; }
@keyframes pop{ from{ transform:scale(.98); opacity:.0 } to{ transform:scale(1); opacity:1 } }
.modal-head{ padding:12px 14px; display:flex; align-items:center; justify-content:space-between; border-bottom:1px solid var(--line); }
.modal-title{ font-weight:800; }
.iconbtn{ background:transparent; color:var(--muted); border:0; font-size:18px; cursor:pointer; }
.iconbtn:hover{ color:var(--fg); }
.modal-body{ padding:16px; }
.modal-body.grid{ display:grid; gap:12px; grid-template-columns: repeat(2, minmax(0, 1fr)); }
.modal-body label{ display:flex; flex-direction:column; gap:6px; font-size:14px; color:var(--fg); }
.modal-body input[type="text"], .modal-body select{
  height:38px; padding:0 10px; border-radius:10px; border:1px solid var(--line); background:#fff; color:var(--fg); outline:none;
}
.modal-foot{ padding:12px 16px; display:flex; justify-content:flex-end; gap:8px; border-top:1px solid var(--line); }

/* ย่อจำนวนคอลัมน์เมื่อจอแคบลง */
@media (max-width: 1200px){
  .cards{ grid-template-columns: repeat(3, 1fr); }
}
@media (max-width: 900px){
  .cards{ grid-template-columns: repeat(2, 1fr); }
}
@media (max-width: 600px){
  .cards{ grid-template-columns: 1fr; }
}
/* ===== Footer กลางล่าง ตัวอักษรขาว ===== */
.site-footer{
  flex-shrink:0;
  text-align:center;
  background:#0E1420;     /* แถบเข้มเพื่อให้ตัวหนังสือขาวอ่านออก */
  color:#ffffff;
  padding:12px 8px;
  line-height:1.4;
}
.site-footer a{ color:#fff; text-decoration:none; }
.site-footer a:hover{ text-decoration:underline; }
</style>

<script>
const qs  = (s, r=document)=> r.querySelector(s);
const qsa = (s, r=document)=> Array.from(r.querySelectorAll(s));

function openProjectModal(){ qs('#project-modal')?.classList.remove('hidden'); qs('#p-name')?.focus(); }
function closeProjectModal(){ qs('#project-modal')?.classList.add('hidden'); }
document.addEventListener('click', (e)=>{
  const m = qs('#project-modal'); if(!m || m.classList.contains('hidden')) return;
  const card = qs('.modal-card', m); if (card && !card.contains(e.target) && !e.target.closest('.btn')) closeProjectModal();
});
document.addEventListener('keydown', (e)=>{ if(e.key==='Escape') closeProjectModal(); });

const cardsWrap = qs('#project-cards');
let cards = qsa('.card.project', cardsWrap);

const els = {
  q: qs('#q'),
  team: qs('#filter-team'),
  status: qs('#filter-status'),
  sort: qs('#sort'),
  empty: qs('#project-empty'),
  count: qs('#proj-count'),
};
const normalize = s => (s||'').toString().trim().toLowerCase();

function applyFilter(){
  const q = normalize(els.q?.value);
  const team = normalize(els.team?.value);
  const st = normalize(els.status?.value);

  let visible = 0;
  cards.forEach(c=>{
    const name=c.dataset.name, code=c.dataset.code, tm=c.dataset.team, status=c.dataset.status;
    const hitQ = !q || name.includes(q) || code.includes(q) || tm.includes(q);
    const hitTeam = !team || tm===team;
    const hitSt = !st || status===st;
    const show = hitQ && hitTeam && hitSt;
    c.style.display = show ? '' : 'none';
    if (show) visible++;
  });

  els.count.textContent = `ทั้งหมด ${visible} รายการ`;
  els.empty?.classList.toggle('hidden', visible>0);
}

function applySort(){
  const mode = els.sort?.value || 'recent';
  const toNum = s => Number(s||0);
  const sorted = [...cards].sort((a,b)=>{
    if (mode==='name') return a.dataset.name.localeCompare(b.dataset.name);
    if (mode==='team') return a.dataset.team.localeCompare(b.dataset.team) || a.dataset.name.localeCompare(b.dataset.name);
    if (mode==='progress') return toNum(b.dataset.progress) - toNum(a.dataset.progress);
    const ua = Date.parse(a.dataset.updated || '') || 0;
    const ub = Date.parse(b.dataset.updated || '') || 0;
    return ub - ua;
  });
  sorted.forEach(el=> cardsWrap.appendChild(el));
}

['input','change'].forEach(evt=>{
  els.q?.addEventListener(evt, applyFilter);
  els.team?.addEventListener(evt, applyFilter);
  els.status?.addEventListener(evt, applyFilter);
  els.sort?.addEventListener(evt, ()=>{ applySort(); applyFilter(); });
});
applySort(); applyFilter();

function saveProjectMock(){
  const name  = normalize(qs('#p-name')?.value);
  const code  = normalize(qs('#p-code')?.value);
  const team  = normalize(qs('#p-team')?.value);
  const stat  = normalize(qs('#p-status')?.value);
  if (!name || !code || !team) { alert('กรุณากรอก ชื่อ / โค้ด / ทีม'); return; }

  const pct = 0, id = Date.now(), updated = (new Date()).toISOString();
  const a = document.createElement('a');
  const statClass = stat.replace(/\s+/g,'-'); // on hold -> on-hold
  a.className = 'card project';
  a.href = '<?= $BASE ?>/project?id='+id;
  a.dataset.name = name; a.dataset.code = code; a.dataset.team = team; a.dataset.status = stat; a.dataset.progress = pct; a.dataset.updated = updated;
  a.innerHTML = `
    <div class="toprow">
      <div class="title ellip">${escapeHtml(cap(name))}</div>
      <span class="pill ${statClass}">${escapeHtml(cap(stat))}</span>
    </div>
    <div class="meta-row">
      <span class="chip"><b>#${escapeHtml(code.toUpperCase())}</b></span>
      <span class="chip">${escapeHtml(cap(team))}</span>
    </div>
    <div class="progress-wrap">
      <div class="progress"><div class="progress__bar" style="--pct:${pct}%"></div></div>
      <span class="progress__pct">${pct}%</span>
    </div>
  `;
  cardsWrap.prepend(a);
  cards = qsa('.card.project', cardsWrap);
  closeProjectModal();
  applySort(); applyFilter();
}
function escapeHtml(s){ return (s||'').replace(/[&<>"']/g, m=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;' }[m])); }
function cap(s){ s=String(s||''); return s.charAt(0).toUpperCase()+s.slice(1); }
</script>
